
/**
 * @file: mspin_core_adapter.c
 *
 * @version: $Id:$
 *
 * @release: $Name:$
 *
 * MySpin Core Adapter
 *
 * @component: MSPIN
 *
 * @author: Thilo Bjoern Fickel ICT-ADITG/SW2 tfickel@de.adit-jv.com
 *
 * @copyright: (c) 2003 - 2013 ADIT Corporation
 *
 * @history
 * 0.1 TFickel Initial version
 *
 ***********************************************************************/

#include "mspin_core_adapter.h"
#include "mspin_core_callback_adapter.h"
#include "mspin_connection_adapter.h"
#include "mspin_lm_adapter.h"
#include "mspin_logging.h"
#include "mspin_measurement.h"
#include "mySPIN-Core.h"
#include <semaphore.h> // semaphore
#include <pthread.h>   // pthread_create
#include <sys/prctl.h>
#include <errno.h>
#include <string.h>

pthread_t gRunCoreThread;
mspin_context_t* gContext = NULL;
BOOL gStartStopCoreThreadQuit = FALSE;
BOOL gStart = FALSE;
sem_t gStartStopSemaphore;
coreStarted_CB gCoreStartedCB = NULL;
void *gCoreStartedToken = NULL;
#ifndef MSPIN_X86SIM
  static pthread_mutex_t gLMMutex = PTHREAD_MUTEX_INITIALIZER;
#endif

static const char* mspin_core_getCapabilityName(MSPIN_CLIENTCAPABILITIES cap)
{
    switch (cap)
    {
        //Cases must match definition of MSPIN_CLIENTCAPABILITIES
        case MSPIN_CLIENTCAPABILITIES_NONE:
        {
            return "No Capabilities";
            //break;
        }
        case MSPIN_CLIENTCAPABILITIES_INITIATEPHONECALL:
        {
            return "Initiate Phonecall";
            //break;
        }
        case MSPIN_CLIENTCAPABILITIES_NAVIGATETO:
        {
            return "NavigateTo";
            //break;
        }
        case MSPIN_CLIENTCAPABILITIES_SCREENCOPY:
        {
            return "Screencopy";
            //break;
        }
        case MSPIN_CLIENTCAPABILITIES_POSITIONINFORMATION:
        {
            return "Position Information";
            //break;
        }
        case MSPIN_CLIENTCAPABILITIES_AUDIOHANDLING:
        {
            return "Audio Handling";
            //break;
        }
        case MSPIN_CLIENTCAPABILITIES_PACKETEXTENSIONS:
        {
            return "PacketExtensions";
            //break;
        }
        case MSPIN_CLIENTCAPABILITIES_PARTIALFRAMEUPDATE:
        {
            return "PatialFrameUpdate";
            //break;
        }
        case MSPIN_CLIENTCAPABILITIES_NOHOMEBUTTON:
        {
            return "NoHomeButton";
            //break;
        }
        case MSPIN_CLIENTCAPABILITIES_REQUIRESFOCUSCONTROL:
        {
            return "RequiresFocusControl";
            //break;
        }
        case MSPIN_CLIENTCAPABILITIES_ISTWOWHEELER:
        {
            return "IsTwoWheeler";
            //break;
        }
        case MSPIN_CLIENTCAPABILITIES_ISOTHERVEHICLE:
        {
            return "IsOtherVehicle";
            //break;
        }
        case MSPIN_CLIENTCAPABILITIES_ALL:
        {
            return "All";
            //break;
        }
        default:
        {
            return "Unknown capability";
            //break;
        }
    }
}

static void* mspin_core_startStop(void* param)
{
    MSPIN_ERROR ret = MSPIN_ERROR_UNKNOWN;

    MSPIN_UNUSED(param);

    prctl(PR_SET_NAME,"myspinCore",0,0,0);

    while (!gStartStopCoreThreadQuit)
    {
        (void) sem_wait(&gStartStopSemaphore); // wait till event comes in

        if (gStart)
        {
            //Start mySPIN Core
            if (gContext)
            {
                ret = mspin_core_start(gContext);
                if (MSPIN_SUCCESS != ret)
                {
                    mspin_log_printLn(eMspinVerbosityError,
                            "%s() ERROR: Failed to start core with rc=%d", __FUNCTION__, ret);
                }

                if (gCoreStartedCB)
                {
                    gCoreStartedCB(gCoreStartedToken, ret);
                }
            }
            else
            {
                mspin_log_printLn(eMspinVerbosityFatal, "%s() FATAL ERROR: gContext not set",
                        __FUNCTION__);
                return NULL;
            }
        }
        else
        {
            //Stop mySPIN Core
            if (gContext)
            {
                mspin_log_printLn(eMspinVerbosityInfo, "%s() -> stopping core", __FUNCTION__);
                void* pCore = mspin_core_stopCoreInstance(gContext);
                mspin_core_deleteCoreInstance(gContext, pCore);
                //gContext->pCoreInstance is already set to NULL in 'mspin_core_stopCoreInstance'
            }
            else
            {
                mspin_log_printLn(eMspinVerbosityError,
                        "%s() ERROR: Context not set", __FUNCTION__);
            }
        }
    }

    pthread_exit(NULL);

    return NULL;
}

#ifndef MSPIN_X86SIM
static void* mspin_core_LMCreator(void* param)
{
    //Set thread name
    (void)prctl(PR_SET_NAME,"mspin_LMCreator", 0, 0, 0);

    mspin_context_t* pMspinContext = (mspin_context_t*)param;
    if (!pMspinContext)
    {
        mspin_log_printLn(eMspinVerbosityFatal,
                "%s(context=%p)[%lu] FATAL ERROR: Context is NULL -> exit",
                __FUNCTION__, pMspinContext, (unsigned long int)pthread_self());
        pthread_exit(NULL);
        return NULL;
    }

    if (-1 == sem_init(&(pMspinContext->LMDeletionLock), 0, 0))
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(context=%p)[%lu] ERROR: Failed to init deletion semaphore with %s -> unlock and exit",
                __FUNCTION__, pMspinContext, (unsigned long int)pthread_self(), strerror(errno));
        sem_post(&(pMspinContext->LMCreatedLock)); //increment creation semaphore before leaving
        pMspinContext->LMCreatorThread = 0;
        pthread_exit(NULL);
        return NULL;
    }

    mspin_log_printLn(eMspinVerbosityDebug, "%s(context=%p)[%lu] started",
            __FUNCTION__, pMspinContext, (unsigned long int)pthread_self());

    MSPIN_ERROR result = mspin_lm_createContext(pMspinContext->pLayerManagerContext, pMspinContext->pCoreInstance);

    if (MSPIN_SUCCESS != result)
    {
        mspin_log_printLn(eMspinVerbosityFatal, "%s(context=%p)[%lu] ERROR: Failed to create LM context with %s (%d)",
                __FUNCTION__, pMspinContext, (unsigned long int)pthread_self(), MSPIN_GetErrorName(result), result);
        sem_post(&(pMspinContext->LMCreatedLock));
    }
    else
    {
        pMspinContext->LMReady = TRUE;
        mspin_log_printLn(eMspinVerbosityDebug, "%s(context=%p)[%lu] LM context created and ready to use",
                __FUNCTION__, pMspinContext, (unsigned long int)pthread_self());
        sem_post(&(pMspinContext->LMCreatedLock));

        //continue in case of success with the deletion
        while ((-1 == sem_wait(&(pMspinContext->LMDeletionLock))) && (EINTR == errno))
        {
            mspin_log_printLn(eMspinVerbosityWarn,
                    "%s(context=%p)[%lu] WARNING: Waiting for LMDeletionLock interrupted -> restart",
                    __FUNCTION__, pMspinContext, (unsigned long int)pthread_self());
            continue; /* Restart if interrupted by handler */
        }

        mspin_log_printLn(eMspinVerbosityDebug, "%s(context=%p)[%lu] delete LM context",
                __FUNCTION__, pMspinContext, (unsigned long int)pthread_self());

        int rc = pthread_mutex_lock(&(pMspinContext->LMCtxMutex));
        if (0 != rc)
        {
            mspin_log_printLn(eMspinVerbosityFatal, "%s(context=%p)[%lu] FATAL ERROR: Failed to lock LMCtxMutex with rc=%d",
                    __FUNCTION__, pMspinContext, (unsigned long int)pthread_self(), rc);
        }
        mspin_lm_deleteContext(pMspinContext->pLayerManagerContext);
        rc = pthread_mutex_unlock(&(pMspinContext->LMCtxMutex));
        if (0 != rc)
        {
            mspin_log_printLn(eMspinVerbosityFatal, "%s(context=%p)[%lu] FATAL ERROR: Failed to unlock LMCtxMutex with rc=%d",
                    __FUNCTION__, pMspinContext, (unsigned long int)pthread_self(), rc);
        }

        pMspinContext->LMReady = FALSE;

        mspin_log_printLn(eMspinVerbosityDebug, "%s(context=%p)[%lu] everything done -> terminate thread",
                __FUNCTION__, pMspinContext, (unsigned long int)pthread_self());
    }

    //Reset thread ID to mark the thread as terminated before releasing the semaphore to make sure that other
    // threads waiting for semaphore 'LMDeletedLock' can check when retrieving the semaphore that the thread is
    // dead
    pMspinContext->LMCreatorThread = 0;

    sem_destroy(&(pMspinContext->LMDeletionLock));

    if (-1 == sem_post(&(pMspinContext->LMDeletedLock)))
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(context=%p)[%lu] ERROR: Failed to unlock deleted semaphore with %s",
                __FUNCTION__, pMspinContext, (unsigned long int)pthread_self(), strerror(errno));
    }

    pthread_exit(NULL);

    return NULL;
}
#endif

void mspin_core_asyncStart(mspin_context_t* pContext, coreStarted_CB coreStartedCB, void* token)
{
    if (!pContext)
    {
        mspin_log_printLn(eMspinVerbosityFatal,
                "%s(context=%p) FATAL ERROR: Context is NULL -> leave", __FUNCTION__, pContext);
        return;
    }

    mspin_log_printLn(eMspinVerbosityDebug, "%s(context=%p) called", __FUNCTION__, pContext);

    gStartStopCoreThreadQuit = FALSE;

    (void)sem_init(&gStartStopSemaphore, 0, 0); // lock binary semaphore
    gStart = TRUE;
    gContext = pContext;
    gCoreStartedCB = coreStartedCB;
    gCoreStartedToken = token;

    if (0 == pthread_create(&gRunCoreThread, NULL, mspin_core_startStop, NULL))
    {
        mspin_log_printLn(eMspinVerbosityDebug, "%s(context=%p) startStopCore thread started",
                __FUNCTION__, pContext);
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError,
                "%s(context=%p) ERROR: Could not start mySPIN core thread",
                __FUNCTION__, pContext);
    }

    sem_post(&gStartStopSemaphore); //release semaphore
}

void mspin_core_asyncStop(mspin_context_t* pContext)
{
    gStartStopCoreThreadQuit = TRUE;

    //reset values
    gCoreStartedCB = NULL;
    gCoreStartedToken = NULL;
    gStart = FALSE;
    gContext = pContext;

    //assume thread is already running
    sem_post(&gStartStopSemaphore); //release semaphore

    //Join thread
    pthread_join(gRunCoreThread, NULL);
}

MSPIN_ERROR mspin_core_start(mspin_context_t* pContext)
{
    Flag enable = eFLAG_TRUE;
    S32 errorNumber = 0;

    if (!pContext)
    {
         mspin_log_printLn(eMspinVerbosityFatal,
                 "%s(context=%p){} FATAL ERROR: pContext is NULL",
                 __FUNCTION__, pContext);
         return MSPIN_ERROR_GENERAL;    /* ======== leave function ======== */
    }

    if (!pContext->pLayerManagerContext)
    {
        mspin_log_printLn(eMspinVerbosityFatal,
                "%s(context=%p){} FATAL ERROR: LM context is NULL",
                __FUNCTION__, pContext);
        return MSPIN_ERROR_GENERAL;    /* ======== leave function ======== */
    }

    // delete possible old mySPIN Core instance (to avoid error message check if pCoreInstance is present before)
    if (pContext->pCoreInstance)
    {
        mspin_core_deleteCoreInstance(pContext, pContext->pCoreInstance);
        //'mspin_core_deleteCoreInstance' sets pContext->pCoreInstance to NULL
    }

    //Reset values
    pContext->isCanceled = FALSE; //make sure canceled flag is reset
    pContext->cancelReason = MSPIN_ERROR_UNKNOWN;
    pContext->hasEverBeenReady = FALSE;
    pContext->frameTransmissionActive = FALSE;

    if (pContext->pingTimeout == 0)
    {
        enable = eFLAG_FALSE;
    }

    void *pCoreHandle = mySpin_CreateInstance(pContext, "MySpin Host");
    // create core instance
    if (pCoreHandle == NULL)
    {
        mspin_log_printLn(eMspinVerbosityError,
                "%s(context=%p){%d} mySpin_CreateInstance returned NULL!",
                __FUNCTION__, pContext, pContext->instanceId);
        return MSPIN_ERROR_CORE;    /* ======== leave function ======== */
    }
    pContext->pCoreInstance = pCoreHandle;

    mySpin_SetCallbackContext(pCoreHandle, pContext);

    mySpin_SetTraceOutput(pCoreHandle, mspin_core_onTraceOutput);

    /* PRQA: Lint Message 64: here we need to forward a bit field. API is not correct */
    /*lint -save -e64*/
    mySpin_SetTraceClass(pCoreHandle, 0xF);     // trace up to verbose
    /*lint -restore*/
    mySpin_SetErrorCallback(pCoreHandle, mspin_core_onError);
    mySpin_SetProtocolCallback(pCoreHandle, mspin_core_onProtocol);

    mySpin_SetReceiverLink(pCoreHandle, mspin_conn_receive);
    mySpin_SetSenderLink(pCoreHandle, mspin_conn_send);

    mySpin_SetFrameUpdateStartCallback(pCoreHandle, mspin_core_onFrameUpdateStart);
    mySpin_SetFrameUpdateCallback(pCoreHandle, mspin_core_onFrameRectUpdate);
    mySpin_SetFrameUpdateEndCallback(pCoreHandle, mspin_core_onFrameUpdateEnd);

    mySpin_SetInitiatePhoneCallCallback(pCoreHandle, mspin_core_onInitiatePhoneCall);
    mySpin_SetPhonecallStatusCallback(pCoreHandle, mspin_core_onPhoneCallStatus);
    mySpin_SetAppTransitionStatusCallback(pCoreHandle, mspin_core_onAppTransitionStatus);
    mySpin_SetAppInactiveCallback(pCoreHandle, mspin_core_onAppInactive);

#if (MSPIN_IVI_CORE_VERSION < 0x01020000UL) //Not supported anymore since IVI Core 1.2
    mySpin_SetBlockStatusCallback(pCoreHandle, mspin_core_onBlockStatus);
#else //Supported since IVI Core 1.2
    mySpin_SetLauncherStateCallback(pCoreHandle, mspin_core_onLauncherState);
    mySpin_SetAppListChangedCallback(pCoreHandle, mspin_core_onAppListChanged);
    mySpin_SetAppIconResponseCallback(pCoreHandle, mspin_core_onAppIconResponse);
    mySpin_SetAppStartedResponseCallback(pCoreHandle, mspin_core_onAppStartedResponse);
#endif // (MSPIN_IVI_CORE_VERSION < 0x01020000UL)

#if (MSPIN_IVI_CORE_VERSION >= 0x01010000UL) //Since IVI Core 1.1
    mySpin_SetNavigateToCallback(pCoreHandle, mspin_core_onNavigateTo);
    mySpin_SetCustomDataStringCallback(pCoreHandle, mspin_core_onCustomString);
    mySpin_SetCustomDataIntCallback(pCoreHandle, mspin_core_onCustomInt);
    mySpin_SetVehicleDataRequestCallback(pCoreHandle, mspin_core_onVehicleDataRequest);
    mySpin_VoiceSupportInfo(pCoreHandle, pContext->voiceSessionSupport, pContext->voiceSessionConstraint);
    mySpin_SetVoiceSessionRequestCallback(pCoreHandle, mspin_core_onVoiceSessionRequest);
    mySpin_SetAudioRequestCallback(pCoreHandle, mspin_core_onAudioRequest);
#else
    mySpin_SetCustomMessageCallback(pCoreHandle, mspin_core_onCustomString);
#endif // (MSPIN_IVI_CORE_VERSION >= 0x01010000UL)

#if (MSPIN_IVI_CORE_VERSION > 0x01020300UL) //Supported since IVI Core 1.2.3
    mySpin_SetPTTAvailableCallback(pCoreHandle, mspin_core_onPTTAvailable);
#endif //(MSPIN_IVI_CORE_VERSION > 0x01020300UL)

    mySpin_SetOnMainThreadStartCallback(pCoreHandle, mspin_core_onMainThreadStart);
    mySpin_SetOnMainThreadEndCallback(pCoreHandle, mspin_core_onMainThreadEnd);

    if (pContext->OnFrameUpdateRawCB)
    {
        mySpin_SetFrameUpdateRawCallback(pCoreHandle, mspin_core_onFrameUpdateRaw);
        pContext->useRawOutput = eFLAG_TRUE; // raw will be used automatically, if CB is set
    }
    mySpin_SetFrameOutputType(pCoreHandle, pContext->useRawOutput);
    mySpin_SetPixelFormat(pCoreHandle, (PixelFormat)pContext->pLayerManagerContext->pixelFormat, pContext->pLayerManagerContext->bytesPerPixel, pContext->pixelEndianess);
#if (MSPIN_IVI_CORE_VERSION >= 0x01010000UL) //Since IVI Core 1.1
    mySpin_SetFrameSize(pCoreHandle, pContext->pLayerManagerContext->frameWidth,
            pContext->pLayerManagerContext->frameHeight,
            pContext->physicalWidth, pContext->physicalHeight);

#if (MSPIN_IVI_CORE_VERSION < 0x01010200UL) //Till IVI Core 1.2.1
    //Overriding compression when preferred compression is set
    if (MSPIN_FRAME_COMPRESSION_NONE != pContext->preferredCompression)
    {
        if (MSPIN_FRAME_COMPRESSION_JPEG == pContext->preferredCompression)
        {
            mspin_log_printLn(eMspinVerbosityInfo, "%s(pContext=%p){%d} Overriding compression with JPEG (till Core 1.2.1)",
                  __FUNCTION__, pContext, pContext->instanceId);
            mySpin_OverrideCompressionFormat(pCoreHandle, eFRAMECOMPRESSION_JPEG);
        }
        else if (MSPIN_FRAME_COMPRESSION_ZLIB == pContext->preferredCompression)
        {
            mspin_log_printLn(eMspinVerbosityInfo, "%s(pContext=%p){%d} Overriding compression with ZLIB (till Core 1.2.1)",
                  __FUNCTION__, pContext, pContext->instanceId);
            mySpin_OverrideCompressionFormat(pCoreHandle, eFRAMECOMPRESSION_ZLIB);
        }
        else if (MSPIN_FRAME_COMPRESSION_UNCOMPRESSED == pContext->preferredCompression)
        {
            mspin_log_printLn(eMspinVerbosityInfo, "%s(pContext=%p){%d} Overriding compression with Uncompressed (till Core 1.2.1)",
                  __FUNCTION__, pContext, pContext->instanceId);
            mySpin_OverrideCompressionFormat(pCoreHandle, eFRAMECOMPRESSION_Uncompressed);
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityError,
                  "%s(pContext=%p){%d} ERROR: preferred compression format (%d) is not supported => no override (till Core 1.2.1)",
                  __FUNCTION__, pContext, pContext->instanceId, (int)pContext->preferredCompression);
        }
    }
#else //(MSPIN_IVI_CORE_VERSION >= 0x01010200UL) //since IVI Core 1.2.2
    if (MSPIN_FRAME_COMPRESSION_NONE != pContext->preferredCompression)
    {
        if (MSPIN_FRAME_COMPRESSION_JPEG == pContext->preferredCompression)
        {
            mspin_log_printLn(eMspinVerbosityInfo, "%s(pContext=%p){%d} Limit supported compressions to JPEG (since Core 1.2.2)",
                  __FUNCTION__, pContext, pContext->instanceId);
            mySpin_SetSupportedCompressions(pCoreHandle, (UInt32)eFRAMECOMPRESSION_JPEG);
        }
        else if (MSPIN_FRAME_COMPRESSION_ZLIB == pContext->preferredCompression)
        {
            mspin_log_printLn(eMspinVerbosityInfo, "%s(pContext=%p){%d} Limit supported compressions to ZLIB (since Core 1.2.2)",
                  __FUNCTION__, pContext, pContext->instanceId);
            mySpin_SetSupportedCompressions(pCoreHandle, (UInt32)eFRAMECOMPRESSION_ZLIB);
        }
        else if (MSPIN_FRAME_COMPRESSION_UNCOMPRESSED == pContext->preferredCompression)
        {
            mspin_log_printLn(eMspinVerbosityInfo, "%s(pContext=%p){%d} Limit supported compressions to Uncompressed (since Core 1.2.2)",
                  __FUNCTION__, pContext, pContext->instanceId);
            mySpin_SetSupportedCompressions(pCoreHandle, (UInt32)eFRAMECOMPRESSION_Uncompressed);
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityError,
                  "%s(pContext=%p){%d} ERROR: preferred compression format (%d) is not supported => do nothing (since Core 1.2.2)",
                  __FUNCTION__, pContext, pContext->instanceId, (int)pContext->preferredCompression);
        }
    }
#endif
#else //(MSPIN_IVI_CORE_VERSION < 0x01010000UL)
    mySpin_SetCompressionFormat(pCoreHandle, pContext->compressionFormat);
    mySpin_SetFrameSize(pCoreHandle, pContext->pLayerManagerContext->frameWidth, pContext->pLayerManagerContext->frameHeight);
#endif // (MSPIN_IVI_CORE_VERSION >= 0x01010000UL)

    mySpin_SetInitialisationTimeout(pCoreHandle, pContext->initializationTimeout);

    U32 mask = 1;

    while (mask <= (U32)MSPIN_CLIENTCAPABILITIES_ISOTHERVEHICLE) //must check for highest value. Otherwise higher values will not be set
    {
        //All client client capabilities except MSPIN_CLIENTCAPABILITIES_SCREENCOPY and
        //MSPIN_CLIENTCAPABILITIES_PACKETEXTENSIONS may be altered
        if ((mask != MSPIN_CLIENTCAPABILITIES_SCREENCOPY) && (mask != MSPIN_CLIENTCAPABILITIES_PACKETEXTENSIONS))
        {
            mspin_log_printLn(eMspinVerbosityDebug, "%s(context=%p){%d} %s capability '%s'(%d)",
                    __FUNCTION__, pContext, pContext->instanceId, (mask & pContext->capabilities) ? "enabling" : "disabling",
                    mspin_core_getCapabilityName((MSPIN_CLIENTCAPABILITIES)mask), mask);

            mySpin_SetClientCapabilities(pCoreHandle, mask, (mask & pContext->capabilities) ? eFLAG_TRUE : eFLAG_FALSE);
        }
        mask *= 2;
    }

    //Enable custom keys using a custom key list profile + PTT key
#if (MSPIN_IVI_CORE_VERSION >= 0x01020000UL) //Since IVI Core 1.2
    if ((MSPIN_CUSTOMDATA_KEYPOROFILE_NONE != pContext->keyProfile) || (pContext->customPTTKey))
    {
        eReturnCode coreRC = eRETURN_OKAY;
        const unsigned int maxNumberOfKeys = 7;
        unsigned int numberOfKeys = 0;
        unsigned int nextKeyIndex = 0;
        ClientCustomDataKeyCode keys[maxNumberOfKeys];
        char keyStringList[256] = "";

        //Support PTT key
        if (pContext->customPTTKey)
        {
            numberOfKeys += 1;
            keys[nextKeyIndex++] = eCLIENTCUSTOMDATAKEYCODE_PushToTalk;
            strcat(keyStringList, "PTT");

            if (MSPIN_CUSTOMDATA_KEYPOROFILE_NONE != pContext->keyProfile)
            {
                strcat(keyStringList, ", ");
            }
        }

        //Support key profiles
        if (MSPIN_CUSTOMDATA_KEYPOROFILE_NONE != pContext->keyProfile)
        {
            keys[nextKeyIndex++] = eCLIENTCUSTOMDATAKEYCODE_Ok;
            keys[nextKeyIndex++] = eCLIENTCUSTOMDATAKEYCODE_Back;
            keys[nextKeyIndex++] = eCLIENTCUSTOMDATAKEYCODE_DPadUp;
            keys[nextKeyIndex++] = eCLIENTCUSTOMDATAKEYCODE_DPadDown;

            if (MSPIN_CUSTOMDATA_KEYPOROFILE_6KEYS == pContext->keyProfile)
            {
                keys[nextKeyIndex++] = eCLIENTCUSTOMDATAKEYCODE_DPadLeft;
                keys[nextKeyIndex++] = eCLIENTCUSTOMDATAKEYCODE_DPadRight;

                numberOfKeys += 6;
                strcat(keyStringList, "OK, Back, DPadUp, DPadDown, DPadLeft, DPadRight");
            }
            else if (MSPIN_CUSTOMDATA_KEYPOROFILE_4KEYS == pContext->keyProfile)
            {
                numberOfKeys += 4;
                strcat(keyStringList, "OK, Back, DPadUp, DPadDown");
            }
        }

        coreRC = mySpin_SetCustomKeysSupported(pCoreHandle, numberOfKeys, keys);
        if (eRETURN_OKAY == coreRC)
        {
            mspin_log_printLn(eMspinVerbosityDebug,
                    "%s(context=%p){%d} %u keys enabled (%s)",
                    __FUNCTION__, pContext, pContext->instanceId, numberOfKeys, keyStringList);
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityError,
                    "%s(context=%p){%d} ERROR: Failed to enable %u keys (%s) with %d",
                    __FUNCTION__, pContext, pContext->instanceId, numberOfKeys, keyStringList, coreRC);
        }
    }
#endif //(MSPIN_IVI_CORE_VERSION >= 0x01020000UL)

    (void)sem_init(&(pContext->coreStartingLock), 0, 0);  // binary semaphore locked
    mspin_log_printLn(eMspinVerbosityInfo,
            "%s(context=%p){%d} coreHandle=%p ...",
            __FUNCTION__, pContext, pContext->instanceId, pCoreHandle);
    if (eFLAG_TRUE != mySpin_StartMainThreadEx(pCoreHandle, &errorNumber))
    {
        mspin_log_printLn(eMspinVerbosityError,
                "%s(pContext=%p){%d} ERROR: Failed to start main thread of core with error=%d",
                __FUNCTION__, pContext, pContext->instanceId, errorNumber);
        return MSPIN_ERROR_CORE;
    }
    else
    {
        if (0 == errorNumber)
        {
            mspin_log_printLn(eMspinVerbosityDebug,
                    "%s(pContext=%p){%d} main thread just started",
                    __FUNCTION__, pContext, pContext->instanceId);
        }
        else if (1 == errorNumber)
        {
            mspin_log_printLn(eMspinVerbosityDebug,
                    "%s(pContext=%p){%d} main thread of core was already running",
                    __FUNCTION__, pContext, pContext->instanceId);
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityWarn,
                    "%s(pContext=%p){%d} WARNING: main thread started but error number=%d not correct",
                    __FUNCTION__, pContext, pContext->instanceId, errorNumber);
        }
    }

    //Enable/disable ping watchdog
    if (enable)
    {
        mySpin_EnablePingWatchdog(pCoreHandle, enable, pContext->pingTimeout); //enable
    }
    else
    {
#if (MSPIN_IVI_CORE_VERSION >= 0x01020300UL) //Supported since IVI Core 1.2.3
        //To detect communication errors, it might be helpful to send data. This can be
        // achieved since Core 1.2.3 by enabling watchdog and sending ping ack on detecting watchdog
        // (when enabled in this combination it does not terminate immediately the connection)

        if (pContext->pConnectionHandle && (MSPIN_CONNECTION_AOAP == pContext->pConnectionHandle->connectionType))
        {
            mspin_log_printLn(eMspinVerbosityDebug, "%s(context=%p){%d} Enable sending ping ACKs on 2s watchdog (since Core 1.2.3)",
                    __FUNCTION__, pContext, pContext->instanceId);
            mySpin_EnablePingWatchdog(pCoreHandle, eFLAG_TRUE, 2); //enable
            mySpin_SendPingAckOnWatchdog(pCoreHandle, eFLAG_TRUE);
        }
        else
        {
            mySpin_EnablePingWatchdog(pCoreHandle, enable, pContext->pingTimeout); //disable
        }
#else
        mySpin_EnablePingWatchdog(pCoreHandle, enable, pContext->pingTimeout); //disable
#endif //#if (MSPIN_IVI_CORE_VERSION >= 0x01020300UL)
    }

    (void)sem_wait(&(pContext->coreStartingLock));  // wait until core gets ready
    if (pContext->isCanceled == FALSE)
    {
        mspin_log_printLn(eMspinVerbosityInfo,
                "%s(context=%p){%d} core ready",
                __FUNCTION__, pContext, pContext->instanceId);
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityInfo,
                "%s(context=%p){%d} core start aborted with reason='%s'(%d)",
                __FUNCTION__, pContext, pContext->instanceId, MSPIN_GetErrorName(pContext->cancelReason),
                (int)pContext->cancelReason);
        return pContext->cancelReason; /* ======== leave function ======== */
    }

    //Issue first frame request if not started in suspend mode.
    if (!pContext->isSuspended)
    {
        // first frame request is not incremental (=force)
        pContext->issueFirstFrameRendered = TRUE;
        mspin_core_requestFrameUpdate(pContext, FALSE);
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityWarn,
                "%s(context=%p){%d} WARNING: Skip first frame request because mySPIN is suspended!",
                __FUNCTION__, pContext, pContext->instanceId);
    }

    return MSPIN_SUCCESS;
}

void* mspin_core_stopCoreInstance(mspin_context_t *pContext)
{
    if (!pContext)
    {
        mspin_log_printLn(eMspinVerbosityError,
                "%s(ctx=%p) ERROR: mySPIN context is NULL",
                __FUNCTION__, pContext);
        return NULL;
    }

    if (!pContext->pCoreInstance)
    {
        mspin_log_printLn(eMspinVerbosityDebug,
                "%s(ctx=%p) Core instance is NULL -> nothing to stop",
                __FUNCTION__, pContext);
        return NULL;
    }

    mspin_log_printLn(eMspinVerbosityInfo,
            "%s(ctx=%p) stopping core instance=%p",
            __FUNCTION__, pContext, pContext->pCoreInstance);
    void* pCore = pContext->pCoreInstance;
    pContext->pCoreInstance = NULL; // prevent that some one else calls Core functions
    mySpin_SetSenderLink(pCore, NULL);
    mySpin_SetReceiverLink(pCore, NULL);

    return pCore;
}

void mspin_core_deleteCoreInstance(mspin_context_t *pContext, void *pCore)
{
    if (!pContext)
    {
        mspin_log_printLn(eMspinVerbosityError,
                "%s(ctx=%p) ERROR: mySPIN context is NULL",
                __FUNCTION__, pContext);
        return;
    }

    if (!pCore)
    {
        mspin_log_printLn(eMspinVerbosityError,
                "%s(core=%p) ERROR: pCore is NULL",
                __FUNCTION__, pCore);
        return;
    }

    U32 cnt = 0;
    S32 commCount = 0;
    do
    {
        commCount = mspin_conn_getCommCount(pContext);  // check if Core still uses send or receive
        if (commCount > 0)
        {
            mspin_log_printLn(eMspinVerbosityInfo, "%s(core=%p) Core still in use, commCount=%d, waiting %d ...",
                    __FUNCTION__, pCore, commCount, cnt);
            usleep(200*1000);   // 200ms
        }
    }
    while ((++cnt < 10*5) && (commCount > 0));

    commCount = mspin_conn_getCommCount(pContext);
    if (commCount > 0)
    {
        mspin_log_printLn(eMspinVerbosityWarn, "%s(core=%p) WARNING: Core still in use, commCount=%d - deleting anyway!",
                __FUNCTION__, pCore, commCount);
    }

    mspin_log_printLn(eMspinVerbosityDebug,
            "%s(ctx=%p) before mySpin_DeleteInstance",
            __FUNCTION__, pContext);

    mySpin_DeleteInstance(pCore);

    mspin_log_printLn(eMspinVerbosityDebug,
            "%s(ctx=%p) after mySpin_DeleteInstance => return",
            __FUNCTION__, pContext);
}

void mspin_core_requestFrameUpdate(mspin_context_t *pContext, bool incremental)
{
    if (pContext && pContext->pLayerManagerContext && pContext->pCoreInstance)
    {
        mspin_measure_startNewFrame(); //measure frame latency

        // first frame request not incremental
#if (MSPIN_IVI_CORE_VERSION < 0x01020000UL) //Function renamed to mySpin_FramebufferUpdateRequest since IVI Core 1.2
        mySpin_FrambufferUpdateRequest(pContext->pCoreInstance, 0, 0,
                (UInt16)pContext->pLayerManagerContext->frameWidth, (UInt16)pContext->pLayerManagerContext->frameHeight,
                incremental ? eFLAG_TRUE : eFLAG_FALSE);
#else
        mySpin_FramebufferUpdateRequest(pContext->pCoreInstance, 0, 0,
                (UInt16)pContext->pLayerManagerContext->frameWidth, (UInt16)pContext->pLayerManagerContext->frameHeight,
                incremental ? eFLAG_TRUE : eFLAG_FALSE);
#endif // (MSPIN_IVI_CORE_VERSION < 0x01020000UL)
    }
    else if (!pContext)
    {
        mspin_log_printLn(eMspinVerbosityError,
                "%s(ctx=%p) ERROR: Context is NULL",
                __FUNCTION__, pContext);
    }
    else if (!pContext->pLayerManagerContext)
    {
        mspin_log_printLn(eMspinVerbosityError,
                "%s(ctx=%p) ERROR: LayerManager context is NULL",
                __FUNCTION__, pContext);
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError,
                "%s(ctx=%p) ERROR: Core instance is NULL",
                __FUNCTION__, pContext);
    }
}

MSPIN_ERROR mspin_core_createLMContext(mspin_context_t *pContext)
{
    mspin_log_printLn(eMspinVerbosityDebug, "%s(ctx=%p) entered", __FUNCTION__, pContext);

    if (!pContext)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(ctx=%p) ERROR: pContext is NULL", __FUNCTION__, pContext);
        return MSPIN_ERROR_GENERAL;
    }

    pContext->LMReady = FALSE;

    // On LSIM with EGL we are not allowed to make egl calls from different threads.
    // On target with APX we create/delete the LM context from an extra thread.
#ifdef MSPIN_X86SIM
    MSPIN_ERROR result = mspin_lm_createContext(pContext->pLayerManagerContext, pContext->pCoreInstance);

    if (MSPIN_SUCCESS != result)
    {
        mspin_log_printLn(eMspinVerbosityFatal, "%s(context=%p)[%lu] ERROR: Failed to create LM context with %s (%d)",
                __FUNCTION__, pContext, (unsigned long int)pthread_self(), MSPIN_GetErrorName(result), result);
    }
    else
    {
       pContext->LMReady = TRUE;
    }
#else
    S32 rc = -1;
    pthread_attr_t attr;

    if (0 != pthread_mutex_lock(&gLMMutex))
    {
        mspin_log_printLn(eMspinVerbosityFatal, "%s(ctx=%p) FATAL ERROR: Failed to acquire LM mutex lock",
                __FUNCTION__, pContext);

        /* PRQA: Lint Message 454: A thread mutex isn't locked in this case. So ignore this error */
        /*lint -save -e454*/
        return MSPIN_ERROR_GENERAL;
        /*lint -restore*/
    }

    mspin_log_printLn(eMspinVerbosityDebug, "%s(ctx=%p) lock acquired", __FUNCTION__, pContext);

    if (pContext->LMCreatorThread)
    {
        mspin_log_printLn(eMspinVerbosityError,
                "%s(ctx=%p) ERROR: The creator thread is already running -> cancel LM creation",
                __FUNCTION__, pContext);
        pthread_mutex_unlock(&gLMMutex); //try to unlock LayerManager mutex
        return MSPIN_ERROR_GENERAL;
    }

    if (-1 == sem_init(&(pContext->LMCreatedLock), 0, 0))
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(ctx=%p) ERROR: Failed to init creation semaphore with %s",
                __FUNCTION__, pContext, strerror(errno));
        pthread_mutex_unlock(&gLMMutex); //try to unlock LayerManager mutex
        return MSPIN_ERROR_GENERAL;
    }

    rc = pthread_attr_init(&attr);
    if (0 != rc)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(ctx=%p) ERROR: Failed to init thread attributes with rc=%d",
                __FUNCTION__, pContext, rc);
        pthread_mutex_unlock(&gLMMutex); //try to unlock LayerManager mutex
        return MSPIN_ERROR_GENERAL;
    }

    rc = pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_DETACHED); //in detached mode
    if (0 != rc)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(ctx=%p) ERROR: Failed to set detach state with rc=%d",
                __FUNCTION__, pContext, rc);
        pthread_mutex_unlock(&gLMMutex); //try to unlock LayerManager mutex
        return MSPIN_ERROR_GENERAL;
    }

    rc = pthread_create(&(pContext->LMCreatorThread), &attr, mspin_core_LMCreator, (void*)pContext); //ToDo: There is currently no support for running multiple mySPIN instances
    if (0 == rc)
    {
        mspin_log_printLn(eMspinVerbosityDebug, "%s(ctx=%p) thread id is %d", __FUNCTION__, pContext, pContext->LMCreatorThread);
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(ctx=%p) ERROR: Failed to create thread with rc=%d", __FUNCTION__, pContext, rc);
        pContext->LMCreatorThread = 0;
    }

    (void)pthread_attr_destroy(&attr);

    while ((-1 == sem_wait(&(pContext->LMCreatedLock))) && (EINTR == errno))
    {
        mspin_log_printLn(eMspinVerbosityWarn,
                "%s(ctx=%p) WARNING: Waiting for LMCreatedLock interrupted -> restart",
                __FUNCTION__, pContext);
        continue; /* Restart if interrupted by handler */
    }

    sem_destroy(&(pContext->LMCreatedLock)); //not needed anymore

    pthread_mutex_unlock(&gLMMutex); //try to unlock LayerManager mutex
#endif
    if (pContext->LMReady)
    {
        mspin_log_printLn(eMspinVerbosityDebug, "%s(ctx=%p) LayerManager context created and ready to be used",
                __FUNCTION__, pContext);
        return MSPIN_SUCCESS;
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(ctx=%p) ERROR: Failed to create LayerManager context",
                __FUNCTION__, pContext);
        return MSPIN_ERROR_LAYERMANAGER;
    }
}

void mspin_core_deleteLMContext(mspin_context_t *pContext)
{
    mspin_log_printLn(eMspinVerbosityDebug, "%s(ctx=%p) entered",  __FUNCTION__, pContext);

    if (!pContext)
    {
        mspin_log_printLn(eMspinVerbosityError,
                "%s(ctx=%p) ERROR: Context is NULL",
                __FUNCTION__, pContext);
        return;
    }

#ifdef MSPIN_X86SIM
    mspin_lm_deleteContext(pContext->pLayerManagerContext);
    pContext->LMReady = FALSE;
#else
    if (0 != pthread_mutex_lock(&gLMMutex))
    {
        mspin_log_printLn(eMspinVerbosityFatal,
                "%s(ctx=%p) FATAL ERROR: Failed to acquire LM mutex lock",
                __FUNCTION__, pContext);

        /* PRQA: Lint Message 454: A thread mutex isn't locked in this case. So ignore this error */
        /*lint -save -e454*/
        return;
        /*lint -restore*/
    }

    mspin_log_printLn(eMspinVerbosityDebug, "%s(ctx=%p) LM lock acquired",  __FUNCTION__, pContext);

    if (0 != pContext->LMCreatorThread)
    {
        mspin_log_printLn(eMspinVerbosityDebug, "%s(ctx=%p) destroying LM context...",
                __FUNCTION__, pContext);

        int rc = sem_init(&(pContext->LMDeletedLock), 0, 0);
        if (-1 == rc)
        {
            mspin_log_printLn(eMspinVerbosityError,
                    "%s(ctx=%p) ERROR: Failed to init deleted semaphore with %s (%d)",
                    __FUNCTION__, pContext, strerror(errno), errno);
        }

        if (-1 == sem_post(&(pContext->LMDeletionLock)))
        {
            mspin_log_printLn(eMspinVerbosityError,
                    "%s(ctx=%p) ERROR: Failed to post deletion semaphore with %s",
                    __FUNCTION__, pContext, strerror(errno));
        }

        if (0 == rc)
        {
            while ((-1 == sem_wait(&(pContext->LMDeletedLock))) && (EINTR == errno))
            {
                mspin_log_printLn(eMspinVerbosityWarn,
                        "%s(ctx=%p) WARNING: Waiting for LMDeletedLock interrupted -> restart",
                        __FUNCTION__, pContext);
                continue; /* Restart if interrupted by handler */
            }

            mspin_log_printLn(eMspinVerbosityDebug,
                    "%s(ctx=%p) LM context destroyed",
                    __FUNCTION__, pContext);

            sem_destroy(&(pContext->LMDeletedLock));
        }
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError,
                "%s(ctx=%p) ERROR: The creator thread isn't running -> cancel deletion",
                __FUNCTION__, pContext);
    }

    pthread_mutex_unlock(&gLMMutex); //try to unlock LayerManager mutex
#endif
}
